class DescriptionSetTemplate:
    """ A Description Set Profile consists of one or more description templates 
        and is initialised first. All wikisyntax is stored until a Statement or 
        Description template is added, then the wiki-syntax is added to the template 
        and will be placed on top of the template in the resulting HTML """
    def __init__(self):
        self.DescriptionTemplates = [] #The list of Description Template, the order is the same as they are added
        self.currentStatementTemplate = None
        self.currentDescriptionTemplate = None
        self.wikiSyntax = ""
    
    
    def addDescriptionTemplate(self, descriptionTemplate):
        #Add the description template (unless it is null) and set it as current 
        if not(descriptionTemplate == None):
            self.currentDescriptionTemplate = descriptionTemplate
            self.DescriptionTemplates.append(self.currentDescriptionTemplate)
            #Also need to remove the current statement template since a new Description is beginning
            self.currentStatementTemplate = None
            #Add the wiki-syntax to the template and reset it
            if (len(self.wikiSyntax)>0):
                self.currentDescriptionTemplate.setDescribingWikiText(self.wikiSyntax)
                self.wikiSyntax = ""
            
    def addStatmentTemplate(self, statementTemplate):
        #Add the statement template to the current Description Template and set it as current 
        if not(statementTemplate == None):
            #Cannot add if no Description Template is current
            if not (self.currentDescriptionTemplate == None):
                self.currentStatementTemplate = statementTemplate
                self.currentDescriptionTemplate.addStatementTemplate(self.currentStatementTemplate)
            #Add the wiki-syntax to the template and reset it
            if (len(self.wikiSyntax)>0):
                self.currentStatementTemplate.setDescribingWikiText(self.wikiSyntax)
                self.wikiSyntax = ""
    
    def addNonLiteralConstraint(self, nonLiteralConstraint):
        if not (nonLiteralConstraint == None):
            #Only add if a Statement Template is current, otherwise ignore
            if not(self.currentStatementTemplate == None):
                #Also need to add the wikisyntax
                if (len(self.wikiSyntax)>0):
                    nonLiteralConstraint.setDescribingWikiText(self.wikiSyntax)
                    self.wikiSyntax = ""
                self.currentStatementTemplate.addNLC(nonLiteralConstraint)
                
    def addLiteralConstraint(self, literalConstraint):
        if not (literalConstraint == None):
            #Only add if no Statement Template is current, otherwise ignore
            if not(self.currentStatementTemplate == None):
                #Also need to add the wikisyntax
                if (len(self.wikiSyntax)>0):
                    literalConstraint.setDescribingWikiText(self.wikiSyntax)
                    self.wikiSyntax = ""
                #Add the 
                self.currentStatementTemplate.addLC(literalConstraint)
    
    """ Stores the normal wiki-syntax in between Templates and Constraints"""
    def addNormalWikiSyntax(self, wikiString):
        if(len(self.wikiSyntax)>0):
            self.wikiSyntax += "\r\n"+wikiString
        else:
            self.wikiSyntax = wikiString
    
    """ Returns a string that is the XML-representation of the Description Set template""" 
    def toXMLString(self):
        returnString = "<DescriptionSetTemplate xmlns=\"http://dublincore.org/xml/dc-dsp/2008/01/14\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:schemaLocation=\"http://dublincore.org/xml/dc-dsp/2008/01/14 http://dublincore.org/architecturewiki/DescriptionSetProfile?action=AttachFile&amp;do=get&amp;target=dcmi-dsp.xsd \">\n"
        for x in self.DescriptionTemplates:
            returnString += x.toXMLString()
        return returnString+"</DescriptionSetTemplate>"
    
    """ Returns a string in the form of MoinMoin-wiki-syntax of the Description Set template"""    
    def __str__(self):
        returnString = "" #'''DescriptionSetTemplate'''\r\n"
        for x in self.DescriptionTemplates:
            returnString += x.__str__()
        returnString += self.wikiSyntax
        return returnString
            
class DescriptionTemplate:
    
    """ input is the string for a Description Template, if not parameter is given
        the string is set to the string 'DT()' """
    
    def __init__(self, DTString="DT()"):
        #The minimumOccurrence of this DescriptionTemplate
        self.minOccur = None
    
        #The maximumOccurrence of this DescriptionTemplate
        self.maxOccur = None
    
        #The ID of the Description Template
        self.ID = None
        
        #Variable that indicates if this template is Standalone, possible values are "yes", "no" or "both"
        self.standalone = None
        
        #The list of Resource constraints
        self.ResourceConstraint = []
        
        #The list of statement templates
        self.st = []
        
        #Holder of the wikitext placed above in the resulting HTML
        self.describingWikiText = ""
        
        #Remove starting DT= and surrounding parenthesis
        fixedString = self.fixString(DTString)
        
        #Recursively breking up the different parts
        self.RecursiveSplitString(fixedString)
    
    def RecursiveSplitString(self, fixedString):
        """Find the first '='-sign and split the string 
           into two pieces, one is the string before 
           and the other one after. This corrensponds to attribute and value"""
        triple = partition(fixedString, "=")
        attribute = triple[0].strip()
        rest = triple[2].strip()
        
        if(len(rest)>0):
            spacePartition = partition(rest, " ")
            #Checking the attribute and sets it
            if(attribute.lower().startswith("max")):
                self.setMax(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
                
            elif(attribute.lower().startswith("min")):
                self.setMin(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
                
            elif(attribute.lower().startswith("id")):
                self.setID(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
                
            elif(attribute.lower().startswith("rc")):
                spacePartition = partition(rest, "] ")
                resourceConstraintsString = spacePartition[0]
                if(spacePartition[0].strip().endswith("]")):
                    resourceConstraintsString=resourceConstraintsString[:len(resourceConstraintsString)-1]
                self.setResourceConstraints(resourceConstraintsString)
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
                
            elif(attribute.lower().startswith("standalone")):
                self.setStandalone(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
            
            #This case, attribute not recognized and therefor ignored
            elif(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
        return
    
    def setStandalone(self,givenString):
        self.standalone=removeCitation(givenString)
    
    def setID(self,givenString):
        self.ID=removeCitation(givenString)
    
    def setMax(self,setString):
        #Need to remove citation (but method also works without)
        if(setString.startswith("\"") and
             setString.endswith("\"") and 
             len(setString)>2):
            setString = setString[1:len(setString)-1]
        if(setString.isalnum()):
            self.maxOccur=setString
             
    def setMin(self,setString):
        #Need to remove citation (but method also works without)
        setString = removeCitation(setString)
        if(setString.isalnum()):
            self.minOccur=setString
    
    def fixString(self,stringToFix):
        #Fix so that everything is on one single line
        allLines = stringToFix.splitlines()
        concatinatingString = ""
        for x in allLines:
            concatinatingString += x+" "
        stringToFix=concatinatingString
        
        #Remove any whitespace around
        stringToFix = stringToFix.strip()
        
        #Remove starting and ending parenthesis
        leftParenthesisIndex = stringToFix.find("(")
        if(leftParenthesisIndex < 0):
            return None
        stringToFix = stringToFix[leftParenthesisIndex+1:len(stringToFix)]
        if(stringToFix.endswith(")")):
            #Else assume that it is something that needs to be used
            stringToFix = stringToFix[0:len(stringToFix)-1]
        return stringToFix
    
    def setResourceConstraints(self,setString):
        if(setString.startswith("[")):
           setString = setString[1:len(setString)]
        self.setResourceConstraintsRecursively(setString)
    
    def setResourceConstraintsRecursively(self,setString):
        if(len(setString)>0):
            partitions = partition(setString, ", ")
            rcString = partitions[0].strip()
            self.ResourceConstraint.append(rcString)
            self.setResourceConstraintsRecursively(partitions[2].strip())
    
    """ Returns a string that is the XML-representation of the Descriptions template"""    
    def toXMLString(self):
        returnString = "<DescriptionTemplate"
        #standalone?
        if not(self.ID == None):
            returnString += " ID=\""+self.ID+"\""
        if not(self.maxOccur == None):
            returnString += " maxOccurs=\""+self.maxOccur+"\""
        if not(self.minOccur == None):
            returnString += " minOccurs=\""+self.minOccur+"\""
        if not(self.standalone == None):
            returnString += " standalone=\""+self.standalone+"\""
        returnString += ">\n"
        if(len(self.ResourceConstraint)>0):
            for x in self.ResourceConstraint:
                returnString+="<ResourceClass>"+x+"</ResourceClass>\n"
        if(len(self.st)>0):
            for x in self.st:
                returnString += x.toXMLString()
        return returnString + "</DescriptionTemplate>\n"
    
    def addStatementTemplate(self, statementTemplate):
        self.st.append(statementTemplate)
    
    def setDescribingWikiText(self,string):
        self.describingWikiText = string
    
    """ Returns a string in the form of MoinMoin-wiki-syntax of the Description template"""    
    def __str__(self):
        returnString = self.describingWikiText+"\r\n"
        if(len(self.st)>0):
            for x in self.st:
                returnString += x.__str__()
        return returnString
 
class StatementTemplate:
    
    def __init__(self,string):
        
        #The minimum occurrence of this statement template
        self.minOccur = ""
        
        #The maximum occurrence of this statement template
        self.maxOccur = ""
        
        #The ID of the statement template
        self.ID = ""
        
        #The type of this statement (literal or nonliteral)
        self.type = ""
    
        #Variable  indicates if the statement template is a subproperty
        self.subProperty = False
        
        #The list of property constraint
        self.PropertyConstraints = []
        
        #The list of literal constraint
        self.LiteralConstraints = []
        
        #The list of non-literal constraint
        self.NonLiteralConstraints = []
        
        self.describingWikiText = ""
        
        #Remove starting ST=  and surrounding parenthesis
        fixedString = self.fixString(string)
        
        #Recursively breking up the different parts of this statement template
        self.RecursiveSplitString(fixedString)
    
    def RecursiveSplitString(self, fixedString):
        """Find the first '='-sign and split the string 
           into two pieces, one is the string before 
           and the other one after. This corrensponds to attribute and value"""
        triple = partition(fixedString, "=")
        attribute = triple[0].strip()
        rest = triple[2].strip()
        if(len(rest)>0):
            spacePartition = partition(rest, " ")
            if(attribute.lower().startswith("max")):
                self.setMax(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
            elif(attribute.lower().startswith("min")):
                #spacePartition = partition(rest, " ")
                self.setMin(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
            elif(attribute.lower().startswith("id")):
                #spacePartition = partition(rest," ")
                self.setID(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
            elif(attribute.lower().startswith("type")):
                #spacePartition = partition(rest, " ")
                self.setType(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
            elif(attribute.lower().startswith("pc")):
                if(rest.startswith("{")):
                   spacePartition = partition(rest,"}")
                elif(rest.startswith("\"")):
                    spacePartition = partition(rest,"\"")
                self.setPropertyConstraints(spacePartition[0])
                if(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
            elif(len(spacePartition[2])>0):
                    self.RecursiveSplitString(spacePartition[2])
        return  
    
    def setType(self, string):
        self.type = removeCitation(string)
    
    def setID(self,givenString):
        self.ID=removeCitation(givenString)
    
    def setMax(self,setString):
        #Need to remove citation (but method also works without)
        if(setString.startswith("\"") and
             setString.endswith("\"") and 
             len(setString)>2):
            setString = setString[1:len(setString)-1]
        if(setString.isalnum()):
            self.maxOccur=setString
             
    def setMin(self,setString):
        #Need to remove citation (but method also works without)
        setString = removeCitation(setString)
        if(setString.isalnum()):
            self.minOccur=setString
    
    def fixString(self,stringToFix):
        #Fix so that everything is on one single line
        allLines = stringToFix.splitlines()
        concatinatingString = ""
        for x in allLines:
            concatinatingString += x+" "
        stringToFix=concatinatingString
        
        #Remove any whitespace around
        stringToFix = stringToFix.strip()
        
        #Remove starting and ending parenthesis
        leftParenthesisIndex = stringToFix.find("(")
        if(leftParenthesisIndex < 0):
            return None
        stringToFix = stringToFix[leftParenthesisIndex+1:len(stringToFix)]
        if(stringToFix.endswith(")")):
            #Else assume that it is something that needs to be used
            stringToFix = stringToFix[0:len(stringToFix)-1]
        return stringToFix
    
    def setPropertyConstraints(self, string):
        if(string.startswith("{")): #Assumed to be a set
           string = string[1:len(string)]
           self.setPropertyConstraintsRecursively(string)
        elif(string.startswith("\"")) : #Assumed to be a super-Property
             string = removeCitation(string)
             self.subProperty = True
             self.PropertyConstraints.insert(0,string)
             
    def setPropertyConstraintsRecursively(self,setString):
        if(len(setString)>0):
            partitions = partition(setString, ",")
            rcString = partitions[0].strip()
            self.PropertyConstraints.append(rcString)
            self.setPropertyConstraintsRecursively(partitions[2].strip())
    
    """ Returns a string that is the XML-representation of the Descriptions template"""
    def toXMLString(self):
        returnString="<StatementTemplate"
        if not(self.ID == None or len(self.ID)<1):
            returnString += " ID=\""+self.ID+"\""
        if not(self.minOccur == None or len(self.minOccur)<1):
            returnString += " minOccurs=\""+self.minOccur+"\""
        if not(self.maxOccur == None or len(self.maxOccur)<1):
            returnString += " maxOccurs=\""+self.maxOccur+"\""
        if not(self.type == None or len(self.type)<1):
            returnString += " type=\""+self.type+"\""
        returnString += ">\n"
        if(len(self.PropertyConstraints)>0 ):
            if not self.subProperty:
                for x in self.PropertyConstraints:
                    returnString += "<Property>"+x+"</Property>\n"
            else:
                returnString +="<SubPropertyOf>"+self.PropertyConstraints[0]+"</SubPropertyOf>\n"
            if (len(self.LiteralConstraints)>0):
                for x in self.LiteralConstraints:
                    returnString += x.toXMLString()
            if(len(self.NonLiteralConstraints)>0):
                for x in self.NonLiteralConstraints:
                    returnString += x.toXMLString()
        return returnString + "</StatementTemplate>\n"
    
    def setDescribingWikiText(self,string):
        self.describingWikiText = string
        
    #Adds a Literal Constraint to the Statement Template
    def addLC(self, literalConstraint):
        self.LiteralConstraints.append(literalConstraint)
    
    #Adds a Non Literal Constraint to the Statement Template
    def addNLC(self, nonLiteralConstraint):
        self.NonLiteralConstraints.append(nonLiteralConstraint)
    
    """ Returns a string in the form of MoinMoin-wiki-syntax of the Statement template"""
    def __str__(self):
        returnString= self.describingWikiText+"\r\n"
        if(len(self.PropertyConstraints)>0 ):
            if not self.subProperty:
                for x in self.PropertyConstraints:
                    returnString += "||<rowbgcolor=\"#FFFF99\">Property ||"+x+"||\r\n"
            else:
                returnString +="||<rowbgcolor=\"#FFFFE0\"> Subproperty of|| "+self.PropertyConstraints[0]+"||\r\n"
        #if not(self.ID == None):
        #    returnString += "||<rowbgcolor=\"#FFFF99\"> ID ||"+self.ID+"||\r\n" #Does not need to be displayed
        if not(self.minOccur == None or len(self.minOccur)<1):
            returnString += "||<rowbgcolor=\"#FFFF99\"> Min occurrence||"+self.minOccur+"||\r\n"
        if not(self.maxOccur == None or len(self.maxOccur)<1):
            returnString += "||<rowbgcolor=\"#FFFF99\"> Max occurrence||"+self.maxOccur+"||\r\n"
        if not(self.type == None):
            returnString += "||<rowbgcolor=\"#FFFF99\"> Literal? ||"
            if(self.type.lower().startswith("literal")):
                returnString += "Yes||\r\n"
            elif(self.type.lower().startswith("nonliteral")):
                returnString += "No||\r\n"
            else:
                returnString += "N/A ||\r\n"
        elif(self.type == None):
            returnString += "||<rowbgcolor=\"#FFFF99\"> Literal? || N/A ||\r\n"
        if(len(self.LiteralConstraints)>0):
            for x in self.LiteralConstraints:
                returnString += x.__str__()
            if (self.type.lower().startswith("nonliteral")):
                returnString += "||<rowbgcolor=\"red\"> Warning|| Statement Template of type Literal, cannot have Non-Literal constraints ||\r\n"
        if (len(self.NonLiteralConstraints)>0):
            for x in self.NonLiteralConstraints:
                returnString += x.__str__()
            if (self.type.lower().startswith("literal")):
                returnString += "||<rowbgcolor=\"red\"> Warning|| Statement Template of type Non Literal, cannot have Literal constraints ||\r\n"
        if(len(self.LiteralConstraints)>0 and len(self.NonLiteralConstraints)>0):
            returnString += "||<rowbgcolor=\"red\"> Warning|| Both Non literal and literal constraints given, only one of them is allowes ||\r\n"
        return returnString
        
class LiteralConstraint:
    
    def __init__(self,string):
        
        #The list of languageConstraint
        self.LanguageConstraint = []
        
        #The list Syntax Encoding Scheme constraint
        self.SESConstraint = []
        
        #The list of options 
        self.LiteralConstraintOptions = []
        
        self.describingWikiText = ""
        
        #Remove starting LC=  and surrounding parenthesis
        string = self.fixString(string)
        
        #Recursively breking up the different parts
        self.RecursiveSplitString(string)
    
    """ Returns a string in the form of MoinMoin-wiki-syntax of the Literal constraint"""
    def __str__(self):
        returnString = self.describingWikiText+"\r\n"
        returnString += "||<rowbgcolor=\"skyBlue\">Value (Literal) ||"
        returnString+="[[MiniPage("
        if(len(self.LiteralConstraintOptions)>0):
            returnString += "'''Choose from:'''\\n"
            for x in self.LiteralConstraintOptions:
                tmp = ""
                if(not x[0] == None):
                   if(len(x[0])>0):
                       tmp += "'''Value:'''"+x[0]+" "
                if(not x[1] == None):
                   if(len(x[1])>0):
                       tmp += "'''Language: '''"+x[1]+" "
                if(not x[2] == None):
                   if(len(x[2])>0):
                       tmp += "'''SES:''' "+x[2]+" "
                returnString += "* " + tmp +"\\n"
            returnString +="\\n"
        if(len(self.SESConstraint)>0):
            returnString += "||||'''Syntax Encoding Scheme:'''||\\n"
            if(len(self.SESConstraint)>1):
                if(len(self.SESConstraint[0])>0):
                    returnString += "||'''Occurrence'''||"+self.SESConstraint[0] +"||\\n"
                if(not self.SESConstraint[1] == None):
                    if(len(self.SESConstraint[1])>0):
                        returnString +="||'''Choose from'''||"
                        for x in self.SESConstraint[1]:
                            returnString += x+" [[BR]] "
                        returnString += "||\\n"
            returnString += "\\n"
        if(len(self.LanguageConstraint)>0):
            returnString += "||||'''Language Constraint:'''||\\n"
            if(len(self.LanguageConstraint)>1):
                if(len(self.LanguageConstraint[0])>0):
                    returnString += "||'''Occurrence'''||"+self.LanguageConstraint[0] +"||\\n"
                if(not self.LanguageConstraint[1] ==None):
                    if(len(self.LanguageConstraint[1])>0):
                        returnString += "||'''Choose from'''||"
                        for x in self.LanguageConstraint[1]:
                            returnString += x+" [[BR]] "
                    returnString += "||\r\n"
            returnString += "\\n"
        return returnString + ")]] ||\r\n"
    
    """ Returns a string that is the XML-representation of the Literal constraint"""
    def toXMLString(self):
        returnString = "<LiteralConstraint>"
        if(len(self.LiteralConstraintOptions)>0):
            for x in self.LiteralConstraintOptions:
                tmp = "<LiteralOption"
                if(not x[1] == None):
                   if(len(x[1])>0):
                       tmp += " lang=\""+x[1]+"\""
                if(not x[2] == None):
                   if(len(x[2])>0):
                       tmp += " SES=\""+x[2]+"\""
                tmp += ">"
                if(not x[0] == None):
                   if(len(x[0])>0):
                       tmp += x[0]
                returnString += tmp + "</LiteralOption>\n"
            returnString +="\n"
        if(len(self.LanguageConstraint)>0):
            if(len(self.LanguageConstraint)>1):
                if(len(self.LanguageConstraint[0])>0):
                    returnString += "<LanguageOccurrence>"+self.LanguageConstraint[0] +"</LanguageOccurrence>\n"
                if(not self.LanguageConstraint[1] ==None):
                    if(len(self.LanguageConstraint[1])>0):
                        for x in self.LanguageConstraint[1]:
                            returnString += "<Language>"+x+"</Language>\n"
        if(len(self.SESConstraint)>0):
            if(len(self.SESConstraint)>1):
                if(len(self.SESConstraint[0])>0):
                    returnString += "<SyntaxEncodingSchemeOccurrence>"+self.SESConstraint[0] +"</SyntaxEncodingSchemeOccurrence>\n"
                if(not self.SESConstraint[1] == None):
                    if(len(self.SESConstraint[1])>0):
                        for x in self.SESConstraint[1]:
                            returnString += "<SyntaxEncodingScheme>"+x+"</SyntaxEncodingScheme>\n"
        return returnString +"</LiteralConstraint>\n"
        
    def fixString(self,stringToFix):
        #Fix so that everything is on one single line
        allLines = stringToFix.splitlines()
        concatinatingString = ""
        for x in allLines:
            concatinatingString += x+" "
        stringToFix=concatinatingString
        
        #Remove any whitespace around
        stringToFix = stringToFix.strip()
        
        #Remove starting and ending parenthesis
        leftParenthesisIndex = stringToFix.find("(")
        if(leftParenthesisIndex < 0):
            return None
        stringToFix = stringToFix[leftParenthesisIndex+1:len(stringToFix)]
        if(stringToFix.endswith(")")):
            #Else assume that it is something that needs to be used
            stringToFix = stringToFix[0:len(stringToFix)-1]
        return stringToFix
    
    def RecursiveSplitString(self, string):
        """Find the first '='-sign and split the string 
           into two pieces, one is the string before 
           and the other one after. This corrensponds to attribute and value"""
        string = string.strip()
        if(string.startswith("{")):
           partitions = partition(string, "} ")
           self.setLiteralConstraintOptions(partitions[0])
           self.RecursiveSplitString(partitions[2])
        else:
            triple = partition(string, "=")
            attribute = triple[0].strip()
            rest = triple[2].strip()
            if(len(rest)>0):
                if(attribute.lower().startswith("langc")):
                    if(rest.startswith("(")):
                        secondPartition = partition(rest,")")
                        self.setLanguageConstraint(secondPartition[0])
                        self.RecursiveSplitString(secondPartition[2].strip())
                elif(attribute.lower().startswith("sesconstraint")):
                    if(rest.startswith("(")):
                        secondPartition = partition(rest,")")
                        self.setSESConstraint(secondPartition[0])
                        self.RecursiveSplitString(secondPartition[2])
    
    def setLiteralConstraintOptions(self,string):
        if(string.startswith("{")):
           string = string[1:len(string)]
        self.setLiteralConstraintOptionsRecursively(string)
        
    def setLiteralConstraintOptionsRecursively(self,string):
        if(string.startswith(",")):
           string = string[1:len(string)].strip()
        if(string.startswith("[")):
           string = string[1:len(string)].strip()
           angleBracketPartition = partition(string,"]")
           self.setLiteralConstraintOptionsRecursively(angleBracketPartition[2])
           SESString = angleBracketPartition[0]
           value = ""
           lang = ""
           SES = ""
           while len(SESString)>0:
              equalPartition = partition(SESString,"=")
              setValuePartition = partition(equalPartition[2]," ")
              if(equalPartition[0].lower().startswith("value")):
                  value = setValuePartition[0]
              elif(SESString.lower().startswith("lang")):
                  lang = setValuePartition[0]
              elif(SESString.lower().startswith("ses")):
                  SES = setValuePartition[0]
              SESString = setValuePartition[2]
           tupleToAdd = value, lang, SES
           self.LiteralConstraintOptions.append(tupleToAdd)
    
    def setLanguageConstraint(self,string):
        if(string.startswith("(")):
           string = string[1:len(string)]
        occurrence = ""
        lang = None
        while(len(string)>0):
            if(string.startswith("{")):
                index = string.find("}")
                languages = ""
                if(index>0):
                    languages = string[1:index].strip()
                    if(index<len(string)):
                        string = string[index+1:len(string)]
                    else:
                        string=""
                else:
                    languages = string.strip()
                    string = ""
                lang = languages.split(",")
            elif(string.lower().startswith("occurrence")):
                index = string.find("=")
                if(index>0 and index<len(string)):
                    string = string[index+1:len(string)].strip()
                    partitions = partition(string," ")
                    occurrence = removeCitation(partitions[0])
                    string = partitions[2]
                else:
                    string = ""
            else:
                string = string[1:len(string)]
        self.LanguageConstraint = occurrence, lang

    def setSESConstraint(self,string):
        if(string.startswith("(")):
           string = string[1:]
        occurrence = ""
        SES = None
        while(len(string)>0):
            if(string.startswith("{")):
                index = string.find("}")
                SESes = ""
                if(index>0):
                    SESes = string[1:index].strip()
                    if(index<len(string)):
                        string = string[index+1:]
                    else:
                        string=""
                else:
                    SESes = string.strip()
                    string = ""
                SES = SESes.split(",") 
            elif(string.lower().startswith("occurrence")):
                index = string.find("=")
                if(index>0 and index<len(string)):
                    string = string[index+1:len(string)].strip()
                    partitions = partition(string," ")
                    occurrence = removeCitation(partitions[0])
                    string = partitions[2]
                else:
                    string = ""
            else:
                string = string[1:len(string)] #Needed? For error input that could lock the loop
        self.SESConstraint = occurrence, SES
        
    def setDescribingWikiText(self,string):
        self.describingWikiText = string            
                    
class NonLiteralConstraint:
    
    def __init__(self, string):
        
        #Reference to a description template
        self.Description = ""
        
        self.setConstraints = []
        
        #The list of value URI constraints
        self.valueURIConstraint = []
        
        #The list of Vocabulary Encoding Scheme constraints
        self.VESConstraint = []
        
        #The list of Value String Constraints
        self.valueStringConstraint = []
        
        self.describingWikiText = ""
       
        #Remove starting ST=  and surrounding parenthesis
        string = self.fixString(string)
        
        #Recursively breking up the different parts
        self.RecursiveSplitString(string)
        
    def fixString(self,stringToFix):
        #Fix so that everything is on one single line
        allLines = stringToFix.splitlines()
        concatinatingString = ""
        for x in allLines:
            concatinatingString += x+" "
        stringToFix=concatinatingString
        
        #Remove any whitespace around
        stringToFix = stringToFix.strip()
        
        #Remove starting and ending parenthesis
        leftParenthesisIndex = stringToFix.find("(")
        if(leftParenthesisIndex < 0):
            return None
        stringToFix = stringToFix[leftParenthesisIndex+1:len(stringToFix)]
        if(stringToFix.endswith(")")):
            #Else assume that it is something that needs to be used
            stringToFix = stringToFix[0:len(stringToFix)-1]
        return stringToFix
    
    def RecursiveSplitString(self, string):
        """Find the first '='-sign and split the string 
           into two pieces, one is the string before 
           and the other one after. This corrensponds to attribute and value"""
        string = string.strip()
        if(string.startswith("{")):
           partitions = partition(string, "}") 
           self.setConstraintOptions(partitions[0])
           self.RecursiveSplitString(partitions[2])
        else:
            triple = partition(string, "=")
            attribute = triple[0].strip()
            rest = triple[2].strip()
            if(len(rest)>0):
                if(attribute.lower().startswith("description")):
                    if(rest.startswith("\"")):
                        secondPartition = partition(rest[1:],"\"")
                    else:
                        secondPartition = partition(rest," ")
                    self.setDescription(secondPartition[0])
                    self.RecursiveSplitString(secondPartition[2])
                elif(attribute.lower().startswith("vuriconstraint")):
                    if(rest.startswith("(")):
                        secondPartition = partition(rest,")")
                        self.setVURIConstraint(secondPartition[0])
                        self.RecursiveSplitString(secondPartition[2])
                elif(attribute.lower().startswith("vesconstraint")):
                    if(rest.startswith("(")):
                        secondPartition = partition(rest,")")
                        self.setVESConstraint(secondPartition[0])
                        self.RecursiveSplitString(secondPartition[2])
                elif(attribute.lower().startswith("vstringconstraint")):
                    if(rest.startswith("(")):
                        index = rest.find(")")
                        if(index <0):
                            vString = rest
                        else:
                            vString = rest[0:index+1]
                            restString = rest[index+1:].strip()
                        while(index>0 and not (vString.count("(") == vString.count(")"))):
                              index = restString.find(")")
                              vString += restString[0:index+1]
                              restString = restString[index+1:]
                        self.valueStringConstraint.append(ValueStringConstraint(vString))
                        self.RecursiveSplitString(restString);
                else: #Skip the rest that are inside the parenthesis
                    if(rest.startswith("(")):
                        index = rest.find(")")
                        if(index>0):
                            ignored = rest[:index+1]
                            rest = rest[index+1:]
                            index = rest.find(")")
                            while(index>0 and not ignored.count("(") == ignored.count(")")):
                                ignored = rest[:index]
                                rest = rest[index:]
                                index = rest.find(")")
                                
                    if(len(rest)>0):
                        self.RecursiveSplitString(rest)
                        
    def setVESConstraint(self,string):
        if(string.startswith("(")):
           string = string[1:len(string)]
        occurrence = ""
        uri = None
        while(len(string)>0):
            if(string.startswith("{")):
                index = string.find("}")
                URIs = ""
                if(index>0):
                    URIs = string[1:index].strip()
                    if(index<len(string)):
                        string = string[index+1:len(string)]
                    else:
                        string=""
                else:
                    languages = string.strip()
                    string = ""
                uri = URIs.split(",") 
            elif(string.lower().startswith("occurrence")):
                index = string.find("=")
                if(index>0 and index<len(string)):
                    string = string[index+1:len(string)].strip()
                    partitions = partition(string," ")
                    occurrence = removeCitation(partitions[0]);
                    string = partitions[2]
                else:
                    string = ""
            else:
                string = string[1:len(string)]
        self.VESConstraint = occurrence, uri
        
    def setVURIConstraint(self, string):
        if(string.startswith("(")):
           string = string[1:len(string)]
        occurrence = ""
        uri = None
        while(len(string)>0):
            if(string.startswith("{")):
                index = string.find("}")
                URIs = ""
                if(index>0):
                    URIs = string[1:index].strip()
                    if(index<len(string)):
                        string = string[index+1:len(string)]
                    else:
                        string=""
                else:
                    languages = string.strip()
                    string = ""
                uri = URIs.split(",") 
            elif(string.lower().startswith("occurrence")):
                index = string.find("=")
                if(index>0 and index<len(string)):
                    string = string[index+1:len(string)].strip()
                    partitions = partition(string," ")
                    occurrence = removeCitation(partitions[0]);
                    string = partitions[2]
                else:
                    string = ""
            else:
                string = string[1:len(string)]
        self.valueURIConstraint = occurrence, uri
    
    def setConstraintOptions(self,string):
        if(string.startswith("{")):
           string = string[1:len(string)]
        if(string.endswith("}")):
           string = string[0:len(string)-1]
        self.setConstraintOptionsRecursively(string)
        
    def setConstraintOptionsRecursively(self,setString):
        if(len(setString)>0):
            partitions = partition(setString, ",")
            constraint = partitions[0].strip()
            self.setConstraints.append(constraint);
            self.setConstraintOptionsRecursively(partitions[2].strip())
            
    def setDescription(self,string):
        self.Description = string
        
    def setDescribingWikiText(self,string):
        self.describingWikiText = string
    
    """ Returns a string in the form of MoinMoin-wiki-syntax of the Non Literal constraint"""
    def __str__(self):
        returnString = self.describingWikiText + "\r\n"
        returnString+="||<rowbgcolor=\"skyBlue\"> Value (Non-Literal) ||"
        returnString+="[[MiniPage("
        if(len(self.Description)>0 ):
            returnString += "'''Description''': " + self.Description + "\\n"
        if(len(self.setConstraints)>0):
            returnString += "'''Choose from''':\\n"
            for x in self.setConstraints:
                returnString +="*"+ x +"\\n"
            returnString += "\\n"
        if(len(self.valueURIConstraint)>1):
            returnString += "||||'''Value URI Constraint''': ||\\n"
            returnString += "||'''Occurrence'''|| "+ self.valueURIConstraint[0]+ " ||\\n"
            if(not self.valueURIConstraint[1] == None):
                returnString += "||'''Choose from''':||"
                for x in self.valueURIConstraint[1]:
                    returnString += x+" [[BR]] "
                returnString += "||\\n"
            returnString += "\\n"
        if(len(self.VESConstraint)>1):
            returnString += "||||'''Vocabulary Encoding Scheme Constraint'''||\\n"
            returnString += "||'''Occurrence''': ||"+ self.VESConstraint[0]+ "||\\n"
            if(not self.VESConstraint[1] == None):
                returnString += "||'''Choose from''': ||"
                for x in self.VESConstraint[1]:
                    returnString += x+" [[BR]] "
                returnString += "||\\n"
            returnString += "\\n"
        if(len(self.valueStringConstraint)>0):
            for x in self.valueStringConstraint:
                returnString += x.__str__()
        returnString += ")]]||\r\n"
        return returnString
    
    """ Returns a string that is the XML-representation of the Non Literal constraint"""
    def toXMLString(self):
        returnString="<NonLiteralConstraint"
        if(len(self.Description)>0 ):
            returnString += " descriptionTemplateRef=\"" + self.Description +"\""
        returnString +=">\n"
        if(len(self.setConstraints)>0):
            for x in self.setConstraints:
                returnString +="<ValueClass>"+ x +"</ValueClass>\n"
        if(len(self.valueURIConstraint)>0):
            returnString += "<ValueURIOccurrence>"+ self.valueURIConstraint[0]+ "</ValueURIOccurrence>\n"
            if(len(self.valueURIConstraint)>1):
                if(not self.valueURIConstraint[1] == None):
                    for x in self.valueURIConstraint[1]:
                        returnString += "<ValueURI>"+x+"</ValueURI>\n"
        if(len(self.VESConstraint)>0):
            returnString += "<VocabularyEncodingSchemeOccurrence>"+ self.VESConstraint[0]+ "</VocabularyEncodingSchemeOccurrence>\n"
            if(len(self.VESConstraint)>1):
                if(not self.VESConstraint[1] == None):
                    for x in self.VESConstraint[1]:
                        returnString += "<VocabularyEncodingSchemeURI>"+x+"</VocabularyEncodingSchemeURI>"
        if(len(self.valueStringConstraint)>0):
            for x in self.valueStringConstraint:
                returnString += x.toXMLString()
        returnString += "</NonLiteralConstraint>"
        return returnString

class ValueStringConstraint:
    
    def __init__(self,string):
        
        #The minimum occurence of this constraint
        self.minOccur = None
        
        #The maximum occurence of this constraint
        self.maxOccur = None
        
        #The maximum occurence of this constraint
        self.LanguageConstraint = []
        
        self.SESConstraint = []
        
        self.ConstraintOptions = []
        #Remove starting ST=  and surrounding parenthesis
        string = self.fixString(string)
        
        #Recursively breking up the different parts
        self.RecursiveSplitString(string)
    
    """ Returns a string in the form of MoinMoin-wiki-syntax of the Value String Constraint"""    
    def __str__(self):
        returnString = "||||'''Value String Constraint:'''||\\n"
        if not (self.minOccur == None or len(self.minOccur)<1):
            returnString += "||Min occurrence||"+self.minOccur+"||\\n"
        if not (self.maxOccur == None or len(self.maxOccur)<1):
            returnString += "||Max occurrence||"+self.maxOccur+"||\\n"
        if(len(self.ConstraintOptions)>0):
            returnString += "||''Choose from'':||"
            for x in self.ConstraintOptions:
                tmp = ""
                if(len(x[0])>0):
                    tmp += "''Value'': " + x[0]+" "
                if(len(x[1])>0):
                    tmp += "''Language'': " +x[1]+" "
                if(len(x[2])>0):
                    tmp += "''SES'': "+x[2]
                if(len(tmp)>0):
                    returnString += tmp +"[[BR]]"
            returnString += "||\\n"
        if(len(self.SESConstraint)>1):
            returnString += "||||''Syntax Encoding Syntax Constraint'':||\\n"
            returnString += "||'''Occurrence''': ||"+ self.SESConstraint[0]+ "||\\n"
            if(not self.SESConstraint[1] == None):
                returnString += "||'''Option''':||"
                for x in self.SESConstraint[1]:
                    returnString += x+" [[BR]] "
                returnString += "||\\n"
        if(len(self.LanguageConstraint)>1):
            returnString += "||||''Language Constraint'':||\\n"
            returnString += "||'''Occurrence'''||"+ self.LanguageConstraint[0]+ "||\\n"
            if(not self.LanguageConstraint[1] == None):
                returnString += "||'''Languages'''||"
                for x in self.SESConstraint[1]:
                    returnString += x+" [[BR]] "
                returnString += "||\\n"
        return returnString +"\\n"
    
    """ Returns a string that is the XML-representation of the Value String Constraint""" 
    def toXMLString(self):
        returnString = "<ValueStringConstraint"
        if(not self.minOccur == None):
            returnString += " minOccurs=\""+self.minOccur+"\""
        if(not self.maxOccur == None):
            returnString += " maxOccurs=\""+self.maxOccur+"\""
        returnString += ">"
        if(len(self.ConstraintOptions)>0):
            for x in self.ConstraintOptions:
                tmp = "<LiteralOption"
                if(len(x[1])>0):
                    tmp += " lang=\"" +x[1]+"\""
                if(len(x[2])>0):
                    tmp += " SES=\""+x[2]+"\""
                tmp+=">"
                if(len(x[0])>0):
                    tmp += x[0]
                returnString += tmp +"</LiteralOption>\n"
        if(len(self.LanguageConstraint)>0):
            returnString += "<LanguageOccurrence>"+ self.LanguageConstraint[0]+ "</LanguageOccurrence>\n"
            if(len(self.LanguageConstraint)>1):
                if(not self.LanguageConstraint[1] == None):
                    for x in self.SESConstraint[1]:
                       returnString += "<Language>"+x+"</Language>\n"
        if(len(self.SESConstraint)>0):
            returnString += "<SyntaxEncodingSchemeOccurrence>"+ self.SESConstraint[0]+ "</SyntaxEncodingSchemeOccurrence>\n"
            if(len(self.SESConstraint)>1):
                if(not self.SESConstraint[1] == None):
                    for x in self.SESConstraint[1]:
                        returnString += "<SyntaxEncodingScheme>"+x+"</SyntaxEncodingScheme>\n"
        return returnString +"</ValueStringConstraint>"
    
    def fixString(self,stringToFix):
        #Fix so that everything is on one single line
        allLines = stringToFix.splitlines()
        concatinatingString = ""
        for x in allLines:
            concatinatingString += x+" "
        stringToFix=concatinatingString
        
        #Remove any whitespace around
        stringToFix = stringToFix.strip()
        
        #Remove starting and ending parenthesis
        leftParenthesisIndex = stringToFix.find("(")
        if(leftParenthesisIndex < 0):
            return None
        stringToFix = stringToFix[leftParenthesisIndex+1:len(stringToFix)]
        if(stringToFix.endswith(")")):
            #Else assume that it is something that needs to be used
            stringToFix = stringToFix[0:len(stringToFix)-1]
        return stringToFix
    
    def RecursiveSplitString(self, string):
        string = string.strip()
        if(string.startswith("{")):
           partitions = partition(string, "} ")
           self.setConstraintOptions(partitions[0])
           self.RecursiveSplitString(partitions[2])
        else:
            triple = partition(string, "=")
            attribute = triple[0].strip()
            rest = triple[2].strip()
            if(len(rest)>0):
                if(attribute.lower().startswith("langc")):
                    if(rest.startswith("(")):
                        secondPartition = partition(rest,")")
                        self.setLanguageConstraint(secondPartition[0])
                        self.RecursiveSplitString(secondPartition[2].strip())
                elif(attribute.lower().startswith("min")):
                    if(rest.startswith("\"")):
                        secondPartition = partition(rest[1:],"\"")
                    else:
                        secondPartition = partition(rest," ")
                    self.setMin(secondPartition[0])
                    self.RecursiveSplitString(secondPartition[2])
                elif(attribute.lower().startswith("max")):
                    if(rest.startswith("\"")):
                        secondPartition = partition(rest[1:],"\"")
                    else:
                        secondPartition = partition(rest," ")
                    self.setMax(secondPartition[0])
                    self.RecursiveSplitString(secondPartition[2])
                elif(attribute.lower().startswith("sesconstraint")):
                    if(rest.startswith("(")):
                        secondPartition = partition(rest,")")
                        self.setSESConstraint(secondPartition[0])
                        self.RecursiveSplitString(secondPartition[2])
    
    def setConstraintOptions(self,string):
        if(string.startswith("{")):
           string = string[1:len(string)]
        self.setConstraintOptionsRecursively(string)
        
    def setConstraintOptionsRecursively(self,string):
        if(string.startswith(",")):
           string = string[1:len(string)].strip()
        if(string.startswith("[")):
           string = string[1:len(string)].strip()
           angleBracketPartition = partition(string,"]")
           self.setConstraintOptionsRecursively(angleBracketPartition[2])
           SESString = angleBracketPartition[0]
           value = ""
           lang = ""
           SES = ""
           while len(SESString)>0:
              equalPartition = partition(SESString,"=")
              setValuePartition = partition(equalPartition[2]," ")
              if(equalPartition[0].lower().startswith("value")):
                  value = removeCitation(setValuePartition[0])
              elif(SESString.lower().startswith("lang")):
                  lang = removeCitation(setValuePartition[0])
              elif(SESString.lower().startswith("ses")):
                  SES = removeCitation(setValuePartition[0])
              SESString = setValuePartition[2]
           tupleToAdd = value, lang, SES
           self.ConstraintOptions.append(tupleToAdd)
    
    def setLanguageConstraint(self,string):
        if(string.startswith("(")):
           string = string[1:len(string)]
        occurrence = ""
        lang = None
        while(len(string)>0):
            if(string.startswith("{")):
                index = string.find("}")
                languages = ""
                if(index>0):
                    languages = string[1:index].strip()
                    if(index<len(string)):
                        string = string[index+1:len(string)]
                    else:
                        string=""
                else:
                    languages = string.strip()
                    string = ""
                lang = languages.split(",")
            elif(string.lower().startswith("occurrence")):
                index = string.find("=")
                if(index>0 and index<len(string)):
                    string = string[index+1:len(string)].strip()
                    partitions = partition(string," ")
                    occurrence = removeCitation(partitions[0])
                    string = partitions[2]
                else:
                    string = ""
            else:
                string = string[1:len(string)]
        self.LanguageConstraint = occurrence, lang
        
    def setMin(self,string):
        string = removeCitation(string)
        self.minOccur = string
    
    def setMax(self,string):
        string = removeCitation(string)
        self.maxOccur = string

    def setSESConstraint(self,string):
        if(string.startswith("(")):
           string = string[1:]
        occurrence = ""
        SES = None
        while(len(string)>0):
            if(string.startswith("{")):
                index = string.find("}")
                SESes = ""
                if(index>0):
                    SESes = string[1:index].strip()
                    if(index<len(string)):
                        string = string[index+1:]
                    else:
                        string=""
                else:
                    SESes = string.strip()
                    string = ""
                SES = SESes.split(",")
            elif(string.lower().startswith("occurrence")):
                index = string.find("=")
                if(index>0 and index<len(string)):
                    string = string[index+1:len(string)].strip()
                    partitions = partition(string," ")
                    occurrence = removeCitation(partitions[0])
                    string = partitions[2]
                else:
                    string = ""
            else:
                string = string[1:len(string)] #Needed? For error input that could lock the loop
        self.SESConstraint = occurrence, SES
        
""" Splits a string into three parts where first part is the string until the token
    second part is the token and last part is the string after the token """        
        
def partition(string, token):
    if (string == None ):
        return None
    if ( token == None or string.find(token)<0 ):
        return string, "", ""
    else:
        index = string.find(token)
        return string[:index], string[index:index+len(token)] , string[index+len(token):len(string)]

""" Takes a string as argument and returns the same string with leading and ending citation removed"""
def removeCitation(string):
    if(string==None):
        return None
    returnString = string
    if(returnString.startswith("\"")):
        returnString = returnString[1:len(returnString)]
    if(returnString.endswith)("\""):
        returnString = returnString[0:len(returnString)-1]
    return returnString
